<?php

namespace App\Http\Controllers;

use App\Models\Notification;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class NotificationController extends Controller
{
    public function index()
    {
        Log::info('Web: Loading notifications index');
        $notifications = Notification::with('user')->get(); // Fetch all notifications
        return view('notifications', compact('notifications'));
    }

    public function markAsRead($id)
    {
        Log::info('Web: Received request to mark notification ' . $id . ' as read, Method: ' . request()->method());
        if (!request()->isMethod('post')) {
            Log::warning('Web: Invalid method for notification ' . $id . ', expected POST, got ' . request()->method());
            return redirect()->back()->with('error', 'Invalid request method. Please use the form.');
        }

        try {
            Log::info('Web: Searching for notification ' . $id);
            $notification = Notification::findOrFail($id);
            Log::info('Web: Found notification ' . $id . ', current Is_read: ' . ($notification->Is_read ? '1' : '0'));

            $notification->Is_read = true;
            $notification->save();
            Log::info('Web: Updated notification ' . $id . ', new Is_read: ' . ($notification->fresh()->Is_read ? '1' : '0'));

            $updatedNotification = Notification::with('user')->findOrFail($id);
            Log::info('Web: Returning updated notification ' . $id . ' with Is_read: ' . ($updatedNotification->Is_read ? '1' : '0'));

            return redirect()->back()->with('success', 'Notification marked as read.')
                ->with('updatedNotification', $updatedNotification);
        } catch (\Exception $e) {
            Log::error('Web: Error marking notification ' . $id . ' as read: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Failed to mark notification as read. Please try again.');
        }
    }

    public function indexApi()
    {
        $notifications = Notification::with('user')->get();
        return response()->json([
            'status' => 'success',
            'data' => $notifications,
        ], 200);
    }

    public function markAsReadApi($id)
    {
        Log::info('API: Received request to mark notification ' . $id . ' as read');
        try {
            $notification = Notification::findOrFail($id);
            Log::info('API: Found notification ' . $id . ', current Is_read: ' . ($notification->Is_read ? '1' : '0'));

            $notification->Is_read = true;
            $notification->save();
            Log::info('API: Updated notification ' . $id . ', new Is_read: ' . ($notification->fresh()->Is_read ? '1' : '0'));

            return response()->json([
                'status' => 'success',
                'message' => 'Notification marked as read',
                'data' => $notification->fresh(),
            ], 200);
        } catch (\Exception $e) {
            Log::error('API: Error marking notification ' . $id . ' as read: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to mark notification as read',
            ], 404);
        }
    }

    public function show($id): JsonResponse
    {
        $notification = Notification::with('user')->findOrFail($id);
        return response()->json([
            'status' => 'success',
            'data' => $notification,
        ], 200);
    }

    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'User_id' => 'required|exists:Users,User_id',
            'Title' => 'required|string|max:255',
            'Message' => 'required|string',
            'Type' => 'sometimes|in:info,warning,success,error',
            'Is_read' => 'sometimes|boolean',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $notification = Notification::create($request->only([
            'User_id',
            'Title',
            'Message',
            'Type',
            'Is_read',
        ]));

        return response()->json([
            'status' => 'success',
            'data' => $notification,
        ], 201);
    }

    public function update(Request $request, $id): JsonResponse
    {
        $notification = Notification::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'User_id' => 'sometimes|exists:Users,User_id',
            'Title' => 'sometimes|string|max:255',
            'Message' => 'sometimes|string',
            'Type' => 'sometimes|in:info,warning,success,error',
            'Is_read' => 'sometimes|boolean',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $notification->update($request->only([
            'User_id',
            'Title',
            'Message',
            'Type',
            'Is_read',
        ]));

        return response()->json([
            'status' => 'success',
            'data' => $notification,
        ], 200);
    }

    public function destroy($id): JsonResponse
    {
        $notification = Notification::findOrFail($id);
        $notification->delete();

        return response()->json([
            'status' => 'success',
            'message' => 'Notification deleted successfully',
        ], 200);
    }
}
